/*
 * Decompiled with CFR 0.152.
 */
package androidx.media3.exoplayer.rtsp.reader;

import androidx.media3.common.ParserException;
import androidx.media3.common.util.Assertions;
import androidx.media3.common.util.Log;
import androidx.media3.common.util.ParsableByteArray;
import androidx.media3.common.util.Util;
import androidx.media3.container.NalUnitUtil;
import androidx.media3.exoplayer.rtsp.RtpPacket;
import androidx.media3.exoplayer.rtsp.RtpPayloadFormat;
import androidx.media3.exoplayer.rtsp.reader.RtpPayloadReader;
import androidx.media3.exoplayer.rtsp.reader.RtpReaderUtils;
import androidx.media3.extractor.ExtractorOutput;
import androidx.media3.extractor.TrackOutput;
import org.checkerframework.checker.nullness.qual.MonotonicNonNull;
import org.checkerframework.checker.nullness.qual.RequiresNonNull;

final class RtpH264Reader
implements RtpPayloadReader {
    private static final String TAG = "RtpH264Reader";
    private static final int MEDIA_CLOCK_FREQUENCY = 90000;
    private static final int FU_PAYLOAD_OFFSET = 2;
    private static final int RTP_PACKET_TYPE_STAP_A = 24;
    private static final int RTP_PACKET_TYPE_FU_A = 28;
    private static final int NAL_UNIT_TYPE_IDR = 5;
    private final ParsableByteArray fuScratchBuffer;
    private final ParsableByteArray nalStartCodeArray = new ParsableByteArray(NalUnitUtil.NAL_START_CODE);
    private final RtpPayloadFormat payloadFormat;
    private @MonotonicNonNull TrackOutput trackOutput;
    private int bufferFlags;
    private long firstReceivedTimestamp;
    private int previousSequenceNumber;
    private int fragmentedSampleSizeBytes;
    private long startTimeOffsetUs;

    public RtpH264Reader(RtpPayloadFormat payloadFormat) {
        this.payloadFormat = payloadFormat;
        this.fuScratchBuffer = new ParsableByteArray();
        this.firstReceivedTimestamp = -9223372036854775807L;
        this.previousSequenceNumber = -1;
    }

    @Override
    public void createTracks(ExtractorOutput extractorOutput, int trackId) {
        this.trackOutput = extractorOutput.track(trackId, 2);
        ((TrackOutput)Util.castNonNull((Object)this.trackOutput)).format(this.payloadFormat.format);
    }

    @Override
    public void onReceivingFirstPacket(long timestamp, int sequenceNumber) {
    }

    @Override
    public void consume(ParsableByteArray data, long timestamp, int sequenceNumber, boolean rtpMarker) throws ParserException {
        int rtpH264PacketMode;
        try {
            rtpH264PacketMode = data.getData()[0] & 0x1F;
        }
        catch (IndexOutOfBoundsException e) {
            throw ParserException.createForMalformedManifest(null, (Throwable)e);
        }
        Assertions.checkStateNotNull((Object)this.trackOutput);
        if (rtpH264PacketMode > 0 && rtpH264PacketMode < 24) {
            this.processSingleNalUnitPacket(data);
        } else if (rtpH264PacketMode == 24) {
            this.processSingleTimeAggregationPacket(data);
        } else if (rtpH264PacketMode == 28) {
            this.processFragmentationUnitPacket(data, sequenceNumber);
        } else {
            throw ParserException.createForMalformedManifest((String)String.format("RTP H264 packetization mode [%d] not supported.", rtpH264PacketMode), null);
        }
        if (rtpMarker) {
            if (this.firstReceivedTimestamp == -9223372036854775807L) {
                this.firstReceivedTimestamp = timestamp;
            }
            long timeUs = RtpReaderUtils.toSampleTimeUs(this.startTimeOffsetUs, timestamp, this.firstReceivedTimestamp, 90000);
            this.trackOutput.sampleMetadata(timeUs, this.bufferFlags, this.fragmentedSampleSizeBytes, 0, null);
            this.fragmentedSampleSizeBytes = 0;
        }
        this.previousSequenceNumber = sequenceNumber;
    }

    @Override
    public void seek(long nextRtpTimestamp, long timeUs) {
        this.firstReceivedTimestamp = nextRtpTimestamp;
        this.fragmentedSampleSizeBytes = 0;
        this.startTimeOffsetUs = timeUs;
    }

    @RequiresNonNull(value={"trackOutput"})
    private void processSingleNalUnitPacket(ParsableByteArray data) {
        int numBytesInData = data.bytesLeft();
        this.fragmentedSampleSizeBytes += this.writeStartCode();
        this.trackOutput.sampleData(data, numBytesInData);
        this.fragmentedSampleSizeBytes += numBytesInData;
        int nalHeaderType = data.getData()[0] & 0x1F;
        this.bufferFlags = RtpH264Reader.getBufferFlagsFromNalType(nalHeaderType);
    }

    @RequiresNonNull(value={"trackOutput"})
    private void processSingleTimeAggregationPacket(ParsableByteArray data) {
        data.readUnsignedByte();
        while (data.bytesLeft() > 4) {
            int nalUnitLength = data.readUnsignedShort();
            this.fragmentedSampleSizeBytes += this.writeStartCode();
            this.trackOutput.sampleData(data, nalUnitLength);
            this.fragmentedSampleSizeBytes += nalUnitLength;
        }
        this.bufferFlags = 0;
    }

    @RequiresNonNull(value={"trackOutput"})
    private void processFragmentationUnitPacket(ParsableByteArray data, int packetSequenceNumber) {
        boolean isLastFuPacket;
        byte fuIndicator = data.getData()[0];
        byte fuHeader = data.getData()[1];
        int nalHeader = fuIndicator & 0xE0 | fuHeader & 0x1F;
        boolean isFirstFuPacket = (fuHeader & 0x80) > 0;
        boolean bl = isLastFuPacket = (fuHeader & 0x40) > 0;
        if (isFirstFuPacket) {
            this.fragmentedSampleSizeBytes += this.writeStartCode();
            data.getData()[1] = (byte)nalHeader;
            this.fuScratchBuffer.reset(data.getData());
            this.fuScratchBuffer.setPosition(1);
        } else {
            int expectedSequenceNumber = RtpPacket.getNextSequenceNumber(this.previousSequenceNumber);
            if (packetSequenceNumber != expectedSequenceNumber) {
                Log.w((String)TAG, (String)Util.formatInvariant((String)"Received RTP packet with unexpected sequence number. Expected: %d; received: %d. Dropping packet.", (Object[])new Object[]{expectedSequenceNumber, packetSequenceNumber}));
                return;
            }
            this.fuScratchBuffer.reset(data.getData());
            this.fuScratchBuffer.setPosition(2);
        }
        int fragmentSize = this.fuScratchBuffer.bytesLeft();
        this.trackOutput.sampleData(this.fuScratchBuffer, fragmentSize);
        this.fragmentedSampleSizeBytes += fragmentSize;
        if (isLastFuPacket) {
            this.bufferFlags = RtpH264Reader.getBufferFlagsFromNalType(nalHeader & 0x1F);
        }
    }

    private int writeStartCode() {
        this.nalStartCodeArray.setPosition(0);
        int bytesWritten = this.nalStartCodeArray.bytesLeft();
        ((TrackOutput)Assertions.checkNotNull((Object)this.trackOutput)).sampleData(this.nalStartCodeArray, bytesWritten);
        return bytesWritten;
    }

    private static int getBufferFlagsFromNalType(int nalType) {
        return nalType == 5 ? 1 : 0;
    }
}

