/*
 * Decompiled with CFR 0.152.
 */
package androidx.media3.datasource.cronet;

import android.net.Uri;
import android.text.TextUtils;
import androidx.annotation.Nullable;
import androidx.media3.common.MediaLibraryInfo;
import androidx.media3.common.util.Assertions;
import androidx.media3.common.util.Clock;
import androidx.media3.common.util.ConditionVariable;
import androidx.media3.common.util.UnstableApi;
import androidx.media3.common.util.Util;
import androidx.media3.datasource.BaseDataSource;
import androidx.media3.datasource.DataSourceException;
import androidx.media3.datasource.DataSpec;
import androidx.media3.datasource.DefaultHttpDataSource;
import androidx.media3.datasource.HttpDataSource;
import androidx.media3.datasource.HttpUtil;
import androidx.media3.datasource.TransferListener;
import androidx.media3.datasource.cronet.ByteArrayUploadDataProvider;
import androidx.media3.datasource.cronet.CronetEngineWrapper;
import com.google.common.base.Ascii;
import com.google.common.base.Predicate;
import com.google.common.primitives.Longs;
import com.google.errorprone.annotations.CanIgnoreReturnValue;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executor;
import org.chromium.net.CronetEngine;
import org.chromium.net.CronetException;
import org.chromium.net.NetworkException;
import org.chromium.net.UploadDataProvider;
import org.chromium.net.UrlRequest;
import org.chromium.net.UrlResponseInfo;

public class CronetDataSource
extends BaseDataSource
implements HttpDataSource {
    @UnstableApi
    public static final int DEFAULT_CONNECT_TIMEOUT_MILLIS = 8000;
    @UnstableApi
    public static final int DEFAULT_READ_TIMEOUT_MILLIS = 8000;
    final UrlRequest.Callback urlRequestCallback;
    private static final int READ_BUFFER_SIZE_BYTES = 32768;
    private final CronetEngine cronetEngine;
    private final Executor executor;
    private final int requestPriority;
    private final int connectTimeoutMs;
    private final int readTimeoutMs;
    private final boolean resetTimeoutOnRedirects;
    private final boolean handleSetCookieRequests;
    @Nullable
    private final String userAgent;
    @Nullable
    private final HttpDataSource.RequestProperties defaultRequestProperties;
    private final HttpDataSource.RequestProperties requestProperties;
    private final ConditionVariable operation;
    private final Clock clock;
    @Nullable
    private Predicate<String> contentTypePredicate;
    private final boolean keepPostFor302Redirects;
    private boolean opened;
    private long bytesRemaining;
    @Nullable
    private UrlRequest currentUrlRequest;
    @Nullable
    private DataSpec currentDataSpec;
    @Nullable
    private ByteBuffer readBuffer;
    @Nullable
    private UrlResponseInfo responseInfo;
    @Nullable
    private IOException exception;
    private boolean finished;
    private volatile long currentConnectTimeoutMs;

    @UnstableApi
    protected CronetDataSource(CronetEngine cronetEngine, Executor executor, int requestPriority, int connectTimeoutMs, int readTimeoutMs, boolean resetTimeoutOnRedirects, boolean handleSetCookieRequests, @Nullable String userAgent, @Nullable HttpDataSource.RequestProperties defaultRequestProperties, @Nullable Predicate<String> contentTypePredicate, boolean keepPostFor302Redirects) {
        super(true);
        this.cronetEngine = (CronetEngine)Assertions.checkNotNull((Object)cronetEngine);
        this.executor = (Executor)Assertions.checkNotNull((Object)executor);
        this.requestPriority = requestPriority;
        this.connectTimeoutMs = connectTimeoutMs;
        this.readTimeoutMs = readTimeoutMs;
        this.resetTimeoutOnRedirects = resetTimeoutOnRedirects;
        this.handleSetCookieRequests = handleSetCookieRequests;
        this.userAgent = userAgent;
        this.defaultRequestProperties = defaultRequestProperties;
        this.contentTypePredicate = contentTypePredicate;
        this.keepPostFor302Redirects = keepPostFor302Redirects;
        this.clock = Clock.DEFAULT;
        this.urlRequestCallback = new UrlRequestCallback();
        this.requestProperties = new HttpDataSource.RequestProperties();
        this.operation = new ConditionVariable();
    }

    @Deprecated
    @UnstableApi
    public void setContentTypePredicate(@Nullable Predicate<String> contentTypePredicate) {
        this.contentTypePredicate = contentTypePredicate;
    }

    @UnstableApi
    public void setRequestProperty(String name, String value) {
        this.requestProperties.set(name, value);
    }

    @UnstableApi
    public void clearRequestProperty(String name) {
        this.requestProperties.remove(name);
    }

    @UnstableApi
    public void clearAllRequestProperties() {
        this.requestProperties.clear();
    }

    @UnstableApi
    public int getResponseCode() {
        return this.responseInfo == null || this.responseInfo.getHttpStatusCode() <= 0 ? -1 : this.responseInfo.getHttpStatusCode();
    }

    @UnstableApi
    public Map<String, List<String>> getResponseHeaders() {
        return this.responseInfo == null ? Collections.emptyMap() : this.responseInfo.getAllHeaders();
    }

    @UnstableApi
    @Nullable
    public Uri getUri() {
        return this.responseInfo == null ? null : Uri.parse((String)this.responseInfo.getUrl());
    }

    @UnstableApi
    public long open(DataSpec dataSpec) throws HttpDataSource.HttpDataSourceException {
        long contentLength;
        long bytesToSkip;
        String contentType;
        UrlRequest urlRequest;
        Assertions.checkNotNull((Object)dataSpec);
        Assertions.checkState((!this.opened ? 1 : 0) != 0);
        this.operation.close();
        this.resetConnectTimeout();
        this.currentDataSpec = dataSpec;
        try {
            this.currentUrlRequest = urlRequest = this.buildRequestBuilder(dataSpec).build();
        }
        catch (IOException e) {
            if (e instanceof HttpDataSource.HttpDataSourceException) {
                throw (HttpDataSource.HttpDataSourceException)e;
            }
            throw new OpenException(e, dataSpec, 2000, 0);
        }
        urlRequest.start();
        this.transferInitializing(dataSpec);
        try {
            boolean connectionOpened = this.blockUntilConnectTimeout();
            IOException connectionOpenException = this.exception;
            if (connectionOpenException != null) {
                String message = connectionOpenException.getMessage();
                if (message != null && Ascii.toLowerCase((String)message).contains("err_cleartext_not_permitted")) {
                    throw new HttpDataSource.CleartextNotPermittedException(connectionOpenException, dataSpec);
                }
                throw new OpenException(connectionOpenException, dataSpec, 2001, CronetDataSource.getStatus(urlRequest));
            }
            if (!connectionOpened) {
                throw new OpenException(new SocketTimeoutException(), dataSpec, 2002, CronetDataSource.getStatus(urlRequest));
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new OpenException(new InterruptedIOException(), dataSpec, 1004, -1);
        }
        UrlResponseInfo responseInfo = (UrlResponseInfo)Assertions.checkNotNull((Object)this.responseInfo);
        int responseCode = responseInfo.getHttpStatusCode();
        Map responseHeaders = responseInfo.getAllHeaders();
        if (responseCode < 200 || responseCode > 299) {
            byte[] responseBody;
            long documentSize;
            if (responseCode == 416 && dataSpec.position == (documentSize = HttpUtil.getDocumentSize((String)CronetDataSource.getFirstHeader(responseHeaders, "Content-Range")))) {
                this.opened = true;
                this.transferStarted(dataSpec);
                return dataSpec.length != -1L ? dataSpec.length : 0L;
            }
            try {
                responseBody = this.readResponseBody();
            }
            catch (IOException e) {
                responseBody = Util.EMPTY_BYTE_ARRAY;
            }
            DataSourceException cause = responseCode == 416 ? new DataSourceException(2008) : null;
            throw new HttpDataSource.InvalidResponseCodeException(responseCode, responseInfo.getHttpStatusText(), (IOException)cause, responseHeaders, dataSpec, responseBody);
        }
        Predicate<String> contentTypePredicate = this.contentTypePredicate;
        if (contentTypePredicate != null && (contentType = CronetDataSource.getFirstHeader(responseHeaders, "Content-Type")) != null && !contentTypePredicate.apply((Object)contentType)) {
            throw new HttpDataSource.InvalidContentTypeException(contentType, dataSpec);
        }
        long l = bytesToSkip = responseCode == 200 && dataSpec.position != 0L ? dataSpec.position : 0L;
        this.bytesRemaining = !CronetDataSource.isCompressed(responseInfo) ? (dataSpec.length != -1L ? dataSpec.length : ((contentLength = HttpUtil.getContentLength((String)CronetDataSource.getFirstHeader(responseHeaders, "Content-Length"), (String)CronetDataSource.getFirstHeader(responseHeaders, "Content-Range"))) != -1L ? contentLength - bytesToSkip : -1L)) : dataSpec.length;
        this.opened = true;
        this.transferStarted(dataSpec);
        this.skipFully(bytesToSkip, dataSpec);
        return this.bytesRemaining;
    }

    @UnstableApi
    public int read(byte[] buffer, int offset, int length) throws HttpDataSource.HttpDataSourceException {
        Assertions.checkState((boolean)this.opened);
        if (length == 0) {
            return 0;
        }
        if (this.bytesRemaining == 0L) {
            return -1;
        }
        ByteBuffer readBuffer = this.getOrCreateReadBuffer();
        if (!readBuffer.hasRemaining()) {
            this.operation.close();
            readBuffer.clear();
            this.readInternal(readBuffer, (DataSpec)Util.castNonNull((Object)this.currentDataSpec));
            if (this.finished) {
                this.bytesRemaining = 0L;
                return -1;
            }
            readBuffer.flip();
            Assertions.checkState((boolean)readBuffer.hasRemaining());
        }
        int bytesRead = (int)Longs.min((long[])new long[]{this.bytesRemaining != -1L ? this.bytesRemaining : Long.MAX_VALUE, readBuffer.remaining(), length});
        readBuffer.get(buffer, offset, bytesRead);
        if (this.bytesRemaining != -1L) {
            this.bytesRemaining -= (long)bytesRead;
        }
        this.bytesTransferred(bytesRead);
        return bytesRead;
    }

    @UnstableApi
    public int read(ByteBuffer buffer) throws HttpDataSource.HttpDataSourceException {
        int copyBytes;
        Assertions.checkState((boolean)this.opened);
        if (!buffer.isDirect()) {
            throw new IllegalArgumentException("Passed buffer is not a direct ByteBuffer");
        }
        if (!buffer.hasRemaining()) {
            return 0;
        }
        if (this.bytesRemaining == 0L) {
            return -1;
        }
        int readLength = buffer.remaining();
        if (this.readBuffer != null && (copyBytes = CronetDataSource.copyByteBuffer(this.readBuffer, buffer)) != 0) {
            if (this.bytesRemaining != -1L) {
                this.bytesRemaining -= (long)copyBytes;
            }
            this.bytesTransferred(copyBytes);
            return copyBytes;
        }
        this.operation.close();
        this.readInternal(buffer, (DataSpec)Util.castNonNull((Object)this.currentDataSpec));
        if (this.finished) {
            this.bytesRemaining = 0L;
            return -1;
        }
        Assertions.checkState((readLength > buffer.remaining() ? 1 : 0) != 0);
        int bytesRead = readLength - buffer.remaining();
        if (this.bytesRemaining != -1L) {
            this.bytesRemaining -= (long)bytesRead;
        }
        this.bytesTransferred(bytesRead);
        return bytesRead;
    }

    @UnstableApi
    public synchronized void close() {
        if (this.currentUrlRequest != null) {
            this.currentUrlRequest.cancel();
            this.currentUrlRequest = null;
        }
        if (this.readBuffer != null) {
            this.readBuffer.limit(0);
        }
        this.currentDataSpec = null;
        this.responseInfo = null;
        this.exception = null;
        this.finished = false;
        if (this.opened) {
            this.opened = false;
            this.transferEnded();
        }
    }

    @UnstableApi
    @Nullable
    protected UrlRequest getCurrentUrlRequest() {
        return this.currentUrlRequest;
    }

    @UnstableApi
    @Nullable
    protected UrlResponseInfo getCurrentUrlResponseInfo() {
        return this.responseInfo;
    }

    @UnstableApi
    protected UrlRequest.Builder buildRequestBuilder(DataSpec dataSpec) throws IOException {
        UrlRequest.Builder requestBuilder = this.cronetEngine.newUrlRequestBuilder(dataSpec.uri.toString(), this.urlRequestCallback, this.executor).setPriority(this.requestPriority).allowDirectExecutor();
        HashMap requestHeaders = new HashMap();
        if (this.defaultRequestProperties != null) {
            requestHeaders.putAll(this.defaultRequestProperties.getSnapshot());
        }
        requestHeaders.putAll(this.requestProperties.getSnapshot());
        requestHeaders.putAll(dataSpec.httpRequestHeaders);
        for (Map.Entry headerEntry : requestHeaders.entrySet()) {
            String key = (String)headerEntry.getKey();
            String value = (String)headerEntry.getValue();
            requestBuilder.addHeader(key, value);
        }
        if (dataSpec.httpBody != null && !requestHeaders.containsKey("Content-Type")) {
            throw new OpenException("HTTP request with non-empty body must set Content-Type", dataSpec, 1004, 0);
        }
        String rangeHeader = HttpUtil.buildRangeRequestHeader((long)dataSpec.position, (long)dataSpec.length);
        if (rangeHeader != null) {
            requestBuilder.addHeader("Range", rangeHeader);
        }
        if (this.userAgent != null) {
            requestBuilder.addHeader("User-Agent", this.userAgent);
        }
        requestBuilder.setHttpMethod(dataSpec.getHttpMethodString());
        if (dataSpec.httpBody != null) {
            requestBuilder.setUploadDataProvider((UploadDataProvider)new ByteArrayUploadDataProvider(dataSpec.httpBody), this.executor);
        }
        return requestBuilder;
    }

    private boolean blockUntilConnectTimeout() throws InterruptedException {
        long now = this.clock.elapsedRealtime();
        boolean opened = false;
        while (!opened && now < this.currentConnectTimeoutMs) {
            opened = this.operation.block(this.currentConnectTimeoutMs - now + 5L);
            now = this.clock.elapsedRealtime();
        }
        return opened;
    }

    private void resetConnectTimeout() {
        this.currentConnectTimeoutMs = this.clock.elapsedRealtime() + (long)this.connectTimeoutMs;
    }

    private void skipFully(long bytesToSkip, DataSpec dataSpec) throws HttpDataSource.HttpDataSourceException {
        if (bytesToSkip == 0L) {
            return;
        }
        ByteBuffer readBuffer = this.getOrCreateReadBuffer();
        try {
            while (bytesToSkip > 0L) {
                this.operation.close();
                readBuffer.clear();
                this.readInternal(readBuffer, dataSpec);
                if (Thread.currentThread().isInterrupted()) {
                    throw new InterruptedIOException();
                }
                if (this.finished) {
                    throw new OpenException(dataSpec, 2008, 14);
                }
                readBuffer.flip();
                Assertions.checkState((boolean)readBuffer.hasRemaining());
                int bytesSkipped = (int)Math.min((long)readBuffer.remaining(), bytesToSkip);
                readBuffer.position(readBuffer.position() + bytesSkipped);
                bytesToSkip -= (long)bytesSkipped;
            }
        }
        catch (IOException e) {
            if (e instanceof HttpDataSource.HttpDataSourceException) {
                throw (HttpDataSource.HttpDataSourceException)e;
            }
            throw new OpenException(e, dataSpec, e instanceof SocketTimeoutException ? 2002 : 2001, 14);
        }
    }

    private byte[] readResponseBody() throws IOException {
        byte[] responseBody = Util.EMPTY_BYTE_ARRAY;
        ByteBuffer readBuffer = this.getOrCreateReadBuffer();
        while (!this.finished) {
            this.operation.close();
            readBuffer.clear();
            this.readInternal(readBuffer, (DataSpec)Util.castNonNull((Object)this.currentDataSpec));
            readBuffer.flip();
            if (readBuffer.remaining() <= 0) continue;
            int existingResponseBodyEnd = responseBody.length;
            responseBody = Arrays.copyOf(responseBody, responseBody.length + readBuffer.remaining());
            readBuffer.get(responseBody, existingResponseBodyEnd, readBuffer.remaining());
        }
        return responseBody;
    }

    private void readInternal(ByteBuffer buffer, DataSpec dataSpec) throws HttpDataSource.HttpDataSourceException {
        ((UrlRequest)Util.castNonNull((Object)this.currentUrlRequest)).read(buffer);
        try {
            if (!this.operation.block((long)this.readTimeoutMs)) {
                throw new SocketTimeoutException();
            }
        }
        catch (InterruptedException e) {
            if (buffer == this.readBuffer) {
                this.readBuffer = null;
            }
            Thread.currentThread().interrupt();
            this.exception = new InterruptedIOException();
        }
        catch (SocketTimeoutException e) {
            if (buffer == this.readBuffer) {
                this.readBuffer = null;
            }
            this.exception = new HttpDataSource.HttpDataSourceException((IOException)e, dataSpec, 2002, 2);
        }
        if (this.exception != null) {
            if (this.exception instanceof HttpDataSource.HttpDataSourceException) {
                throw (HttpDataSource.HttpDataSourceException)this.exception;
            }
            throw HttpDataSource.HttpDataSourceException.createForIOException((IOException)this.exception, (DataSpec)dataSpec, (int)2);
        }
    }

    private ByteBuffer getOrCreateReadBuffer() {
        if (this.readBuffer == null) {
            this.readBuffer = ByteBuffer.allocateDirect(32768);
            this.readBuffer.limit(0);
        }
        return this.readBuffer;
    }

    private static boolean isCompressed(UrlResponseInfo info) {
        for (Map.Entry entry : info.getAllHeadersAsList()) {
            if (!((String)entry.getKey()).equalsIgnoreCase("Content-Encoding")) continue;
            return !((String)entry.getValue()).equalsIgnoreCase("identity");
        }
        return false;
    }

    @Nullable
    private static String parseCookies(@Nullable List<String> setCookieHeaders) {
        if (setCookieHeaders == null || setCookieHeaders.isEmpty()) {
            return null;
        }
        return TextUtils.join((CharSequence)";", setCookieHeaders);
    }

    private static void attachCookies(UrlRequest.Builder requestBuilder, @Nullable String cookies) {
        if (TextUtils.isEmpty((CharSequence)cookies)) {
            return;
        }
        requestBuilder.addHeader("Cookie", cookies);
    }

    private static int getStatus(UrlRequest request) throws InterruptedException {
        final ConditionVariable conditionVariable = new ConditionVariable();
        final int[] statusHolder = new int[1];
        request.getStatus(new UrlRequest.StatusListener(){

            public void onStatus(int status) {
                statusHolder[0] = status;
                conditionVariable.open();
            }
        });
        conditionVariable.block();
        return statusHolder[0];
    }

    @Nullable
    private static String getFirstHeader(Map<String, List<String>> allHeaders, String headerName) {
        List<String> headers = allHeaders.get(headerName);
        return headers != null && !headers.isEmpty() ? headers.get(0) : null;
    }

    private static int copyByteBuffer(ByteBuffer src, ByteBuffer dst) {
        int remaining = Math.min(src.remaining(), dst.remaining());
        int limit = src.limit();
        src.limit(src.position() + remaining);
        dst.put(src);
        src.limit(limit);
        return remaining;
    }

    static {
        MediaLibraryInfo.registerModule((String)"media3.datasource.cronet");
    }

    private final class UrlRequestCallback
    extends UrlRequest.Callback {
        private UrlRequestCallback() {
        }

        public synchronized void onRedirectReceived(UrlRequest request, UrlResponseInfo info, String newLocationUrl) {
            UrlRequest.Builder requestBuilder;
            boolean shouldKeepPost;
            if (request != CronetDataSource.this.currentUrlRequest) {
                return;
            }
            UrlRequest urlRequest = (UrlRequest)Assertions.checkNotNull((Object)CronetDataSource.this.currentUrlRequest);
            DataSpec dataSpec = (DataSpec)Assertions.checkNotNull((Object)CronetDataSource.this.currentDataSpec);
            int responseCode = info.getHttpStatusCode();
            if (dataSpec.httpMethod == 2 && (responseCode == 307 || responseCode == 308)) {
                CronetDataSource.this.exception = (IOException)new HttpDataSource.InvalidResponseCodeException(responseCode, info.getHttpStatusText(), null, info.getAllHeaders(), dataSpec, Util.EMPTY_BYTE_ARRAY);
                CronetDataSource.this.operation.open();
                return;
            }
            if (CronetDataSource.this.resetTimeoutOnRedirects) {
                CronetDataSource.this.resetConnectTimeout();
            }
            boolean bl = shouldKeepPost = CronetDataSource.this.keepPostFor302Redirects && dataSpec.httpMethod == 2 && responseCode == 302;
            if (!shouldKeepPost && !CronetDataSource.this.handleSetCookieRequests) {
                request.followRedirect();
                return;
            }
            String cookieHeadersValue = CronetDataSource.parseCookies((List)info.getAllHeaders().get("Set-Cookie"));
            if (!shouldKeepPost && TextUtils.isEmpty((CharSequence)cookieHeadersValue)) {
                request.followRedirect();
                return;
            }
            urlRequest.cancel();
            DataSpec redirectUrlDataSpec = !shouldKeepPost && dataSpec.httpMethod == 2 ? dataSpec.buildUpon().setUri(newLocationUrl).setHttpMethod(1).setHttpBody(null).build() : dataSpec.withUri(Uri.parse((String)newLocationUrl));
            try {
                requestBuilder = CronetDataSource.this.buildRequestBuilder(redirectUrlDataSpec);
            }
            catch (IOException e) {
                CronetDataSource.this.exception = e;
                return;
            }
            CronetDataSource.attachCookies(requestBuilder, cookieHeadersValue);
            CronetDataSource.this.currentUrlRequest = requestBuilder.build();
            CronetDataSource.this.currentUrlRequest.start();
        }

        public synchronized void onResponseStarted(UrlRequest request, UrlResponseInfo info) {
            if (request != CronetDataSource.this.currentUrlRequest) {
                return;
            }
            CronetDataSource.this.responseInfo = info;
            CronetDataSource.this.operation.open();
        }

        public synchronized void onReadCompleted(UrlRequest request, UrlResponseInfo info, ByteBuffer buffer) {
            if (request != CronetDataSource.this.currentUrlRequest) {
                return;
            }
            CronetDataSource.this.operation.open();
        }

        public synchronized void onSucceeded(UrlRequest request, UrlResponseInfo info) {
            if (request != CronetDataSource.this.currentUrlRequest) {
                return;
            }
            CronetDataSource.this.finished = true;
            CronetDataSource.this.operation.open();
        }

        public synchronized void onFailed(UrlRequest request, UrlResponseInfo info, CronetException error) {
            if (request != CronetDataSource.this.currentUrlRequest) {
                return;
            }
            if (error instanceof NetworkException && ((NetworkException)error).getErrorCode() == 1) {
                CronetDataSource.this.exception = new UnknownHostException();
            } else {
                CronetDataSource.this.exception = (IOException)error;
            }
            CronetDataSource.this.operation.open();
        }
    }

    @UnstableApi
    public static final class OpenException
    extends HttpDataSource.HttpDataSourceException {
        public final int cronetConnectionStatus;

        @Deprecated
        public OpenException(IOException cause, DataSpec dataSpec, int cronetConnectionStatus) {
            super(cause, dataSpec, 2000, 1);
            this.cronetConnectionStatus = cronetConnectionStatus;
        }

        public OpenException(IOException cause, DataSpec dataSpec, int errorCode, int cronetConnectionStatus) {
            super(cause, dataSpec, errorCode, 1);
            this.cronetConnectionStatus = cronetConnectionStatus;
        }

        @Deprecated
        public OpenException(String errorMessage, DataSpec dataSpec, int cronetConnectionStatus) {
            super(errorMessage, dataSpec, 2000, 1);
            this.cronetConnectionStatus = cronetConnectionStatus;
        }

        public OpenException(String errorMessage, DataSpec dataSpec, int errorCode, int cronetConnectionStatus) {
            super(errorMessage, dataSpec, errorCode, 1);
            this.cronetConnectionStatus = cronetConnectionStatus;
        }

        public OpenException(DataSpec dataSpec, int errorCode, int cronetConnectionStatus) {
            super(dataSpec, errorCode, 1);
            this.cronetConnectionStatus = cronetConnectionStatus;
        }
    }

    public static final class Factory
    implements HttpDataSource.Factory {
        @Nullable
        private final CronetEngine cronetEngine;
        private final Executor executor;
        private final HttpDataSource.RequestProperties defaultRequestProperties;
        @Nullable
        private final DefaultHttpDataSource.Factory internalFallbackFactory;
        @Nullable
        private HttpDataSource.Factory fallbackFactory;
        @Nullable
        private Predicate<String> contentTypePredicate;
        @Nullable
        private TransferListener transferListener;
        @Nullable
        private String userAgent;
        private int requestPriority;
        private int connectTimeoutMs;
        private int readTimeoutMs;
        private boolean resetTimeoutOnRedirects;
        private boolean handleSetCookieRequests;
        private boolean keepPostFor302Redirects;

        public Factory(CronetEngine cronetEngine, Executor executor) {
            this.cronetEngine = (CronetEngine)Assertions.checkNotNull((Object)cronetEngine);
            this.executor = executor;
            this.defaultRequestProperties = new HttpDataSource.RequestProperties();
            this.internalFallbackFactory = null;
            this.requestPriority = 3;
            this.connectTimeoutMs = 8000;
            this.readTimeoutMs = 8000;
        }

        @Deprecated
        @UnstableApi
        public Factory(CronetEngineWrapper cronetEngineWrapper, Executor executor) {
            this.cronetEngine = cronetEngineWrapper.getCronetEngine();
            this.executor = executor;
            this.defaultRequestProperties = new HttpDataSource.RequestProperties();
            this.internalFallbackFactory = new DefaultHttpDataSource.Factory();
            this.connectTimeoutMs = 8000;
            this.readTimeoutMs = 8000;
        }

        @CanIgnoreReturnValue
        @UnstableApi
        public final Factory setDefaultRequestProperties(Map<String, String> defaultRequestProperties) {
            this.defaultRequestProperties.clearAndSet(defaultRequestProperties);
            if (this.internalFallbackFactory != null) {
                this.internalFallbackFactory.setDefaultRequestProperties(defaultRequestProperties);
            }
            return this;
        }

        @CanIgnoreReturnValue
        @UnstableApi
        public Factory setUserAgent(@Nullable String userAgent) {
            this.userAgent = userAgent;
            if (this.internalFallbackFactory != null) {
                this.internalFallbackFactory.setUserAgent(userAgent);
            }
            return this;
        }

        @CanIgnoreReturnValue
        @UnstableApi
        public Factory setRequestPriority(int requestPriority) {
            this.requestPriority = requestPriority;
            return this;
        }

        @CanIgnoreReturnValue
        @UnstableApi
        public Factory setConnectionTimeoutMs(int connectTimeoutMs) {
            this.connectTimeoutMs = connectTimeoutMs;
            if (this.internalFallbackFactory != null) {
                this.internalFallbackFactory.setConnectTimeoutMs(connectTimeoutMs);
            }
            return this;
        }

        @CanIgnoreReturnValue
        @UnstableApi
        public Factory setResetTimeoutOnRedirects(boolean resetTimeoutOnRedirects) {
            this.resetTimeoutOnRedirects = resetTimeoutOnRedirects;
            return this;
        }

        @CanIgnoreReturnValue
        @UnstableApi
        public Factory setHandleSetCookieRequests(boolean handleSetCookieRequests) {
            this.handleSetCookieRequests = handleSetCookieRequests;
            return this;
        }

        @CanIgnoreReturnValue
        @UnstableApi
        public Factory setReadTimeoutMs(int readTimeoutMs) {
            this.readTimeoutMs = readTimeoutMs;
            if (this.internalFallbackFactory != null) {
                this.internalFallbackFactory.setReadTimeoutMs(readTimeoutMs);
            }
            return this;
        }

        @CanIgnoreReturnValue
        @UnstableApi
        public Factory setContentTypePredicate(@Nullable Predicate<String> contentTypePredicate) {
            this.contentTypePredicate = contentTypePredicate;
            if (this.internalFallbackFactory != null) {
                this.internalFallbackFactory.setContentTypePredicate(contentTypePredicate);
            }
            return this;
        }

        @CanIgnoreReturnValue
        @UnstableApi
        public Factory setKeepPostFor302Redirects(boolean keepPostFor302Redirects) {
            this.keepPostFor302Redirects = keepPostFor302Redirects;
            if (this.internalFallbackFactory != null) {
                this.internalFallbackFactory.setKeepPostFor302Redirects(keepPostFor302Redirects);
            }
            return this;
        }

        @CanIgnoreReturnValue
        @UnstableApi
        public Factory setTransferListener(@Nullable TransferListener transferListener) {
            this.transferListener = transferListener;
            if (this.internalFallbackFactory != null) {
                this.internalFallbackFactory.setTransferListener(transferListener);
            }
            return this;
        }

        @Deprecated
        @CanIgnoreReturnValue
        @UnstableApi
        public Factory setFallbackFactory(@Nullable HttpDataSource.Factory fallbackFactory) {
            this.fallbackFactory = fallbackFactory;
            return this;
        }

        @UnstableApi
        public HttpDataSource createDataSource() {
            if (this.cronetEngine == null) {
                return this.fallbackFactory != null ? this.fallbackFactory.createDataSource() : ((DefaultHttpDataSource.Factory)Assertions.checkNotNull((Object)this.internalFallbackFactory)).createDataSource();
            }
            CronetDataSource dataSource = new CronetDataSource(this.cronetEngine, this.executor, this.requestPriority, this.connectTimeoutMs, this.readTimeoutMs, this.resetTimeoutOnRedirects, this.handleSetCookieRequests, this.userAgent, this.defaultRequestProperties, this.contentTypePredicate, this.keepPostFor302Redirects);
            if (this.transferListener != null) {
                dataSource.addTransferListener(this.transferListener);
            }
            return dataSource;
        }
    }
}

